unit aspxDelphiWCL.AutoPostback;

{
Copyright:  aspxDelphi.net All rights reserved. Use freely.

Zarko Gajic
http://aspxdelphi.net
http://delphi.about.com

}

interface

uses
  aspxDelphiWCL.Designer, System.Collections.Specialized, aspxDelphiWCL.Common,
  System.Web.UI, System.Web.UI.WebControls, System.Text,
  System.ComponentModel, System.ComponentModel.Design, System.Web.UI.Design;

type
  /// Designer for Autopostback
  AutoPostbackDesigner = class(ControlDesigner)
  public
    function GetDesignTimeHtml : string; override;
  end;

  /// <summary>
  /// AutoPostback mimics the TTimer Delphi control found in the VCL.
  /// </summary>
  /// <remarks>AutoPostback has no rendering of its own ... it only emits some client side javascript functions</remarks>
  [
  DefaultProperty('Interval')
  ,DefaultEvent('Postback')
  ,ToolboxData('<{0}:AutoPostback runat=server></{0}:AutoPostback>')
  ,DesignerAttribute(typeof(AutoPostbackDesigner))
  ]
  AutoPostback = class(System.Web.UI.Control, IPostBackEventHandler, IPostBackDataHandler)
  strict private
    FInterval: integer;
    FElapsed: integer;
    FEnabled: boolean;

    FPostback: EventHandler;

    function GetHelperID: string;

    property HelperID : string read GetHelperID;
  strict protected
    procedure OnInit(e : EventArgs); override;
    procedure OnPreRender(e : System.EventArgs); override;
    procedure Render(Output: HtmlTextWriter); override;

    procedure OnPostback(e:EventArgs);
  public
    constructor Create;
    procedure RaisePostBackEvent(eventArgument: string);
    function LoadPostData(postDataKey: string;  postCollection: NameValueCollection): Boolean;
    procedure RaisePostDataChangedEvent;
    procedure Dispose; override;
    procedure SetInterval(const Value: integer);

  published
    [
    Bindable(true)
    ,Category('Behavior')
    ,DefaultValue(1000)
    ,Description('Determines the amount of time, in milliseconds, that passes before the control initiates a post back and raises the Postback event on the server')
    ]
    property Interval: integer read FInterval write SetInterval;
    [
    Bindable(true)
    ,Category('Behavior')
    ,DefaultValue(true)
    ,Description('Controls whether the control will generate Postback event on the server automatically after Interval miliseconds have passed')
    ]
    property Enabled : boolean read FEnabled write FEnabled;

    [
    Browsable(false)
    ,Description('Retrieves the amount of time, in seconds, that have passed before the auto Postback has occured.')
    ]
    property Elapsed : integer read FElapsed;

    //events
    [
    Description('Occurs when Interval miliseconds have passed. Does not get called if Interval <= 0.')
    ]
    property Postback : EventHandler add FPostback remove FPostback;

  end;

implementation

{$REGION 'Constructor, Init, Load'}
constructor AutoPostback.Create;
begin
  inherited;
  FInterval := 1000;
  FElapsed := 0;
  FEnabled := true;

  //Instances := Instances + 1;
end;

procedure AutoPostback.Dispose;
begin
  //Instances := Instances - 1;
  //if Instances < 0 then Instances := 0; //just testing
  inherited;
end;

procedure AutoPostback.OnInit(e: EventArgs);
begin
  inherited;
  if not Common.IsDesignTime(self.Page) then Page.RegisterRequiresPostBack(self);
end;
{$ENDREGION}

{$REGION 'Private properties'}
function AutoPostback.GetHelperID: string;
begin
  result := '__' + ClientID + '_elapsed_';
end;
{$ENDREGION}

{$REGION 'Public properties'}
procedure AutoPostback.SetInterval(const Value: integer);
begin
  if Value < 0 then
    raise ArgumentOutOfRangeException.Create('Interval',Value.ToString, 'Interval property must be >= 0!')
  else
    FInterval := Value;
end;
{$ENDREGION}

{$REGION 'IPostBackDataHandler implementation'}
function AutoPostback.LoadPostData(postDataKey: string; postCollection: NameValueCollection): Boolean;
var
  helper : string;
begin
  helper := postCollection[HelperID];
  if Assigned(helper) then
  begin
    FElapsed := (Interval div 1000) - Convert.ToInt32(helper);
  end;
  result := false;
end;

procedure AutoPostback.RaisePostDataChangedEvent;
begin
  //nothing ... no events to raise
end;
{$ENDREGION}

{$REGION 'IPostBackEventHandler implementation'}
procedure AutoPostback.OnPostback(e: EventArgs);
begin
  if Assigned(FPostback) then FPostback(self, e);
end;

procedure AutoPostback.RaisePostBackEvent(eventArgument: string);
begin
  OnPostback(EventArgs.Empty)
end;
{$ENDREGION}

{$REGION 'Render and related'}
procedure AutoPostback.OnPreRender(e: System.EventArgs);
var
  sb : StringBuilder;
begin
  inherited;

  if not Enabled then Exit;

  If Interval <= 0 then Exit;

  Page.RegisterRequiresPostBack(self);

  sb := StringBuilder.Create();

  //auto-Postback
  sb.Append('<script language="javascript">');
  sb.Append('window.setTimeout("' + self.Page.GetPostBackEventReference(self) +'", ' + Interval.ToString +');');
  sb.Append('</script>');
  if NOT Page.IsClientScriptBlockRegistered(ClientID) then Page.RegisterClientScriptBlock(ClientID,sb.ToString());

  //Elapsed function
  self.Page.RegisterHiddenField(HelperID, Interval.ToString);

  sb.Remove(0,sb.Length);
  sb.Append('<script language="javascript">');
  sb.Append('var aspxdTimer_iTimer = null;');
  sb.Append('var aspxdTimer_iTimerCnt = ' + Convert.ToString(Interval / 1000) + ';');
  sb.Append('var aspxdTimer_ms, aspxdTimer_h;');
  sb.Append('function aspxdTimer()');
  sb.Append('{');
  sb.Append('aspxdTimer_iTimerCnt--; ');
  sb.Append('aspxdTimer_ms = aspxdTimer_iTimerCnt; ');
  sb.Append('aspxdTimer_h = Math.floor(aspxdTimer_ms / 3600); ');
  sb.Append('aspxdTimer_ms = aspxdTimer_ms - aspxdTimer_h * 3600; ');
  sb.Append('document.getElementById("' + HelperID + '").value = aspxdTimer_ms; ');
  //sb.Append('window.status = ms; ');
  sb.Append('aspxdTimer_iTimer = window.setTimeout("aspxdTimer()",1000); ');
  sb.Append('}');
  sb.Append('</script>');
  if NOT Page.IsClientScriptBlockRegistered(HelperID) then Page.RegisterClientScriptBlock(HelperID,sb.ToString());

  //Start Elapsed Timer
  FElapsed := 0;
  sb.Remove(0,sb.Length);
  sb.Append('<script language="JavaScript">');
  sb.Append('aspxdTimer();');
  sb.Append('</script>');
  if NOT Page.IsStartupScriptRegistered(ClientID) then Page.RegisterStartupScript(ClientID,sb.ToString());
end;

procedure AutoPostback.Render(Output: HtmlTextWriter);
begin
  //must be inside form tags
  if not Common.IsDesignTime(Page) then Page.VerifyRenderingInServerForm(self);

  //allow only one AutoPostback per Web page
  //Context.Items collection is a per-request bag of data that is recreated for every request.
  if Context.Items.Contains(typeof(self)) then
    raise Exception.Create('Only one instance of the AutoPostback control can be placed on a Web Form!')
  else
    Context.Items.Add(typeof(self),System.Object(true));
end;
{$ENDREGION}


{$REGION 'AutoPostbackDesigner'}

function AutoPostbackDesigner.GetDesignTimeHtml: string;
begin
  result := self.CreatePlaceHolderDesignTimeHtml();
end;

{$ENDREGION}



end.
